// contacts/app.js — XDock Plus (MV3, no inline JS)

// ====== Utils ======
const DBG = false;
const log = (...a) => DBG && console.log("[contacts]", ...a);

function normPhone(p) {
  const d = String(p || "").replace(/[^\d+]/g, "");
  return d ? (d.startsWith("+") ? d : "+" + d) : "";
}

// Demande au parent (xdock.js) de composer le numéro.
// Si la page contacts est ouverte seule, on tente localement.
function callNumber(phone) {
  const num = normPhone(phone);
  if (!num) return;
  try {
    if (window.parent && window.parent !== window) {
      window.parent.postMessage({ xdockplus: true, type: "CALL", phone: num }, "*");
    } else {
      location.href = "ciscotel:" + num;
    }
  } catch (e) {
    console.warn("callNumber error", e);
  }
}

function alertBox(msg) {
  try { alert("L’extension XDock Plus indique\n\n" + msg); }
  catch { console.warn(msg); }
}

// Petit helper MV3 → background
function sendBg(payload) {
  return new Promise((resolve) => {
    try {
      chrome.runtime.sendMessage(payload, (res) => resolve(res || { ok: false }));
    } catch (e) {
      resolve({ ok: false, error: String(e) });
    }
  });
}

// ====== SVGs (icônes boutons) ======
function phoneSvg() {
  return `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16"
               viewBox="0 0 24 24" fill="currentColor">
            <path d="M6.62 10.79a15.05 15.05 0 0 0 6.59 6.59l2.2-2.2a1 1 0 0 1 1.11-.21c1.21.49 2.53.76 3.88.76a1 1 0 0 1 1 1v3.5a1 1 0 0 1-1 1C10.3 22.13 1.88 13.7 1.88 4a1 1 0 0 1 1-1h3.5a1 1 0 0 1 1 1c0 1.35.26 2.67.76 3.88a1 1 0 0 1-.21 1.11z"/>
          </svg>`;
}
function editSvg() {
  return `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16"
               viewBox="0 0 24 24" fill="currentColor">
            <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04a1.003 1.003 0 0 0 0-1.42l-2.34-2.34a1.003 1.003 0 0 0-1.42 0l-1.83 1.83 3.75 3.75 1.84-1.82z"/>
          </svg>`;
}
function trashSvg() {
  return `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16"
               viewBox="0 0 24 24" fill="currentColor">
            <path d="M16 4V3a2 2 0 0 0-2-2h-4a2 2 0 0 0-2 2v1H3v2h18V4zM8 4V3h8v1zM6 7h12v14H6z"/>
          </svg>`;
}


// ====== État & rendu ======
const EMPTY_DATA = {
  "Matin": [],
  "Après-midi": [],
  "Nuit": [],
  "Prestation": [],
  "Groupage": []
};
let DATA = { ...EMPTY_DATA };
let editingId = null;

function escapeHtml(s) {
  return String(s || "").replace(/[&<>"']/g, m => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' }[m]));
}

function elContact(c) {
  const div = document.createElement("div");
  div.className = "contact";
  div.dataset.id = c.id;

  div.innerHTML = `
    <div class="info">
      <div class="name">${escapeHtml(c.name)}</div>
      <div class="phone">${escapeHtml(c.phone)}</div>
    </div>
    <div class="actions">
      <button type="button" class="icon-btn call" title="Appeler" aria-label="Appeler">
        ${phoneSvg()}
      </button>
      <button type="button" class="icon-btn edit" data-act="edit" title="Modifier" aria-label="Modifier">
        ${editSvg()}
      </button>
      <button type="button" class="icon-btn del" data-act="del" title="Supprimer" aria-label="Supprimer">
        ${trashSvg()}
      </button>
    </div>
  `;

  // Couleurs des boutons (et icônes via currentColor)
  const callBtn = div.querySelector(".icon-btn.call");
  Object.assign(callBtn.style, { background:"#007AFF", color:"#fff", border:"none" });

  const editBtn = div.querySelector(".icon-btn.edit");
  Object.assign(editBtn.style, { background:"#FBBF24", /* jaune */ color:"#111", border:"none" });

  const delBtn = div.querySelector(".icon-btn.del");
  Object.assign(delBtn.style, { background:"#EF4444", /* rouge */ color:"#fff", border:"none" });

  // Actions
  callBtn.onclick = () => callNumber(c.phone);
  editBtn.onclick = () => startEdit(c);
  delBtn.onclick  = () => delContact(c.id);

  return div;
}

function render(filter = "") {
  const f = filter.trim().toLowerCase();
  const mount = (list, id) => {
    const root = document.getElementById(id);
    if (!root) return;
    root.innerHTML = "";
    list.forEach(c => {
      if (!f || (c.name && c.name.toLowerCase().includes(f)) || String(c.phone).includes(f)) {
        root.appendChild(elContact(c));
      }
    });
  };
  mount(DATA["Matin"], "list-matin");
  mount(DATA["Après-midi"], "list-am");
  mount(DATA["Nuit"], "list-nuit");
  mount(DATA["Groupage"], "list-groupage");
  mount(DATA["Prestation"], "list-prestation");
}

// ====== CRUD via background ======
async function load() {
  log("load()");
  const res = await sendBg({ type: "CONTACTS_GET" });
  if (!res?.ok) {
    alertBox("Impossible de charger les contacts (vérifie l’API et le domaine autorisé dans le manifest).");
    return;
  }
  const d = res.data || {};
  // Merge pour s'assurer que tous les groupes existent
  DATA = { ...EMPTY_DATA, ...d };
  render(document.getElementById("search")?.value || "");
}

async function save(contact) {
  log("save()", contact);
  const res = await sendBg({ type: "CONTACTS_POST", body: { action: "upsert", contact } });

  // Transport raté
  if (!res?.ok) {
    alertBox("Erreur sauvegarde (transport).");
    throw new Error("save failed (transport)");
  }

  // Réponse non-JSON ?
  if (!res.data) {
    alertBox("Erreur sauvegarde: http_" + (res.status ?? "200"));
    console.warn("contacts/save response text:", res.text);
    throw new Error("save failed (no json)");
  }

  // Réponse API connue
  if (res.data.ok === true) {
    await load();
    return;
  }

  // Erreurs applicatives
  const err = res.data.error || "inconnue";
  if (err === "duplicate_phone") {
    alertBox("Numéro déjà présent (doublon).");
  } else if (err === "invalid_phone") {
    alertBox("Numéro invalide.");
  } else if (err === "invalid_fields") {
    alertBox("Champs invalides (nom, téléphone ou groupe).");
  } else {
    alertBox("Erreur sauvegarde: " + err);
  }
  throw new Error("save failed (api: " + err + ")");
}


async function delContact(id) {
  if (!confirm("Supprimer ce contact ?")) return;
  const res = await sendBg({ type: "CONTACTS_POST", body: { action: "delete", id } });
  if (!res?.ok || !res.data?.ok) {
    alertBox("Erreur suppression");
    return;
  }
  await load();
}

// ====== Formulaire ======
function startEdit(c) {
  editingId = c.id || null;
  document.getElementById("name").value = c.name || "";
  document.getElementById("phone").value = c.phone || "";
  // c.group peut ne pas être présent si la source est brute : utilise c.grp le cas échéant
  document.getElementById("group").value = c.group || c.grp || "Matin";
  document.getElementById("save").textContent = "Mettre à jour";
}
function resetForm() {
  editingId = null;
  document.getElementById("name").value = "";
  document.getElementById("phone").value = "";
  document.getElementById("group").value = "Matin";
  document.getElementById("save").textContent = "Ajouter";
}

// ====== Init UI ======
(function init() {
  // Bouton Ajouter / MAJ
  document.getElementById("save").onclick = async () => {
    const name = document.getElementById("name").value.trim();
    const phone = document.getElementById("phone").value.trim();
    const group = document.getElementById("group").value;
    if (!name || !phone) { alertBox("Nom et téléphone requis"); return; }
    const contact = { id: editingId, name, phone, group };
    try {
      await save(contact);
      resetForm();
    } catch (e) {
      console.error(e);
    }
  };

  document.getElementById("reset").onclick = resetForm;
  document.getElementById("search").oninput = (e) => render(e.target.value || "");

  load();
})();
